package cn.ninegame.ad.demo.activity;

import android.app.Activity;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.FrameLayout;
import android.widget.TextView;

import androidx.annotation.Nullable;

import java.util.List;

import cn.ninegame.ad.demo.R;
import cn.ninegame.ad.demo.config.NGAdManagerHolder;
import cn.ninegame.ad.demo.util.ToastUtil;
import cn.ninegame.ad.demo.util.UIUtil;
import cn.sirius.nga.ad.NGAdBase;
import cn.sirius.nga.ad.NGAdBase.ExpressAdListener;
import cn.sirius.nga.ad.NGExpressAd;
import cn.sirius.nga.config.AdPlacement;
import cn.sirius.nga.dislike.NGAdDislike;

/**
 * 融合demo，banner广告使用示例。更多功能参考接入文档。
 * <p>
 * 注意1：每次加载的广告，只能展示一次
 * <p>
 * 注意2：banner 的广告容器，需要与广告尺寸的宽高比接近，，不要有遮挡物，否则影响广告结算。
 * <p>
 * 因：部分ADN平台的宽高比限制，默认 banner 的宽高比尺寸为 6.4:1，宽度大于 50px。
 * <p>
 * 如果需要另外的尺寸，请与商务运营同学联系，在后台排除某些ADN源
 * <p>
 * 关于测试：测试广告可能无对应尺寸的广告，测试时没有关系，只要可出广告即可。
 * <p>
 * <p>
 * 接入步骤：
 * 1、创建AdPlacement对象
 * 2、创建NGAdBase对象
 * 3、创建加载、展示监听器
 * 4、加载广告
 * 5、加载成功后，展示广告
 * 6、在onDestroy中销毁广告
 */
public class MediationBannerActivity extends Activity {

    private static final String TAG = "MediationBannerActivity";

    /**
     * 融合广告位（在NG平台的广告位，注意不是adn的代码位）
     */
    public String mMediaId;

    /**
     * Banner广告对象
     */
    private NGExpressAd mBannerAd;

    /**
     * 广告加载监听器
     */
    private ExpressAdListener mBannerListener;

    /**
     * 广告展示监听器
     */
    private NGExpressAd.ExpressAdInteractionListener mBannerInteractionListener;

    /**
     * 接受广告关闭回调
     */
    private NGAdDislike.DislikeInteractionCallback mDislikeCallback;

    /**
     * banner广告容器view
     * <p>
     * 广告容器的宽高比尺寸要与请求的广告宽高比尺寸一致
     * <p>
     * 不要有遮挡物，否则影响广告结算
     * <p>
     * 1. 容器类型不限制；2. 常见错误是 *.LayoutParams 与 容器不匹配，如更改容器请注意检查
     */
    private FrameLayout mBannerContainer;

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.mediation_activity_banner);

        // 融合广告位（在NG平台的广告位，注意不是adn的代码位）
        this.mMediaId = getResources().getString(R.string.banner_media_id);
        TextView tvMediationId = this.findViewById(R.id.tv_media_id);
        tvMediationId.setText(getString(R.string.ad_mediation_id, this.mMediaId));

        // banner广告容器
        mBannerContainer = findViewById(R.id.banner_container);

        // 广告加载
        findViewById(R.id.bt_load).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                loadBannerAd();
            }
        });

        // 广告展示
        findViewById(R.id.bt_show).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                // 不要有遮挡物，否则影响广告结算
                showBannerAd();
            }
        });
    }

    /**
     * 广告加载
     */
    private void loadBannerAd() {
        /** 1、创建AdPlacement对象 */
        AdPlacement adPlacement = new AdPlacement.Builder()
                .setCodeId(this.mMediaId)
                /**
                 * 单位px
                 * 部分ADN平台的宽高比限制，默认 banner 的宽高比尺寸为 6.4:1，宽度大于 50px。
                 */
                .setImageAcceptedSize(
                        /* AcceptedWidth */ UIUtil.dp2px(this, 320f),
                        /* AcceptedHeight */ UIUtil.dp2px(this, 50f)
                )
                .build();

        /** 2、创建NGAdBase对象 */
        NGAdBase adBaseLoader = NGAdManagerHolder.get().createAdNative(this);

        /** 3、创建加载、展示监听器 */
        initListeners();

        /** 4、加载广告 */
        /** 请求广告前，建议判断一次是否有网络，特别是激励广告 */
        adBaseLoader.loadBannerExpressAd(adPlacement, mBannerListener);

        ToastUtil.show(MediationBannerActivity.this, "广告加载中，请稍后");
    }

    /**
     * 5、广告加载成功后，设置监听器，展示广告
     */
    private void showBannerAd() {
        if (mBannerAd != null) {
            /** load成功后，先判断广告是否isReady，如果为false 可间隔一段时间再判断*/
            if (mBannerAd.getMediationManager() != null && mBannerAd.getMediationManager().isReady()) {
                // 广告准备好
                mBannerAd.setExpressInteractionListener(mBannerInteractionListener);
                mBannerAd.setDislikeCallback(this, mDislikeCallback);
                mBannerAd.uploadDislikeEvent("mediation_dislike_event");

                View bannerView = mBannerAd.getExpressAdView();
                if (bannerView != null && mBannerContainer != null) {
                    // 不要有遮挡物，否则影响广告结算
                    mBannerContainer.removeAllViews();
                    mBannerContainer.addView(bannerView);
                }
            } else {
                Log.d(TAG, "广告未准备好，请等待广告准备好后再展示广告");
                // 间隔后重试
            }
        } else {
            Log.d(TAG, "请先加载广告或等待广告加载完毕后再展示广告");
        }
    }

    private void initListeners() {
        // 广告加载监听器
        this.mBannerListener = new ExpressAdListener() {
            @Override
            public void onError(int code, String msg) {
                /** code = 900400、3003：网络不可用，无法连接到服务器；建议引导用户联网后重试。*/
                Log.d(TAG, "banner load fail: errCode: " + code + ", errMsg: " + msg);
                ToastUtil.show(MediationBannerActivity.this, "广告加载失败 onError code = " + code + " msg = " + msg);
            }

            @Override
            public void onExpressAdLoad(List<NGExpressAd> list) {
                if (list != null && list.size() > 0) {
                    Log.d(TAG, "banner load success");
                    /** 获取 Banner广告对象 */
                    mBannerAd = list.get(0);

                    ToastUtil.show(MediationBannerActivity.this, "广告加载成功,可展示");
                } else {
                    Log.d(TAG, "banner load success, but list is null");
                    ToastUtil.show(MediationBannerActivity.this, "广告加载成功,但list为空");
                }
            }
        };
        // 广告展示监听器
        this.mBannerInteractionListener = new NGExpressAd.ExpressAdInteractionListener() {
            @Override
            public void onAdClicked(View view, int i) {
                Log.d(TAG, "banner clicked");
                // 广告被点击
            }

            @Override
            public void onAdShow(View view, int i) {
                Log.d(TAG, "banner showed");
                // 广告被展示
            }

        };
        // dislike监听器，广告关闭时会回调onSelected
        this.mDislikeCallback = new NGAdDislike.DislikeInteractionCallback() {
            @Override
            public void onShow() {
                // 本条 banner 不感兴趣弹窗展示
            }

            @Override
            public void onSelected(int i, String s, boolean b) {
                // 本条 banner广告 被用户请求关闭，移除banner广告
                if (mBannerContainer != null) {
                    mBannerContainer.removeAllViews();
                }
                Log.d(TAG, "banner closed");
            }

            @Override
            public void onCancel() {
                // 本条 banner 不感兴趣弹窗关闭
            }
        };
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        /** 6、在onDestroy中销毁广告 */
        if (mBannerAd != null) {
            mBannerAd.destroy();
        }
    }
}
